<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class SyncStaffData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:sync-staff-data';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        try {
            // Prevent spamming the API: allow once every 60 seconds
            if (RateLimiter::tooManyAttempts('api:get-staff-details', 1)) {
                return response()->json(['error' => 'Too many requests. Please wait.'], 429);
            }

            RateLimiter::hit('api:get-staff-details', 60); // 60 seconds lock

            // Cache to avoid unnecessary API calls
            $data = Cache::remember('staff_api_response', 60, function () {
                $response = Http::retry(3, 5000)
                    ->acceptJson()
                    ->post('https://services.aaft.com/api/get-staff-details', []);

                if (!$response->successful()) {
                    Log::error('API call failed', [
                        'status' => $response->status(),
                        'body' => $response->body()
                    ]);
                    throw new \Exception("API request failed with status: {$response->body()}");
                }

                $decoded = json_decode($response->body(), true);

                // Handle nested JSON strings
                while (is_string($decoded)) {
                    $decoded = json_decode($decoded, true);
                }

                if (!is_array($decoded)) {
                    throw new \Exception("Invalid JSON structure received.");
                }

                return $decoded;
            });

            // Insert data into DB
            foreach ($data as $item) {
                DB::table('staff')->insert(
                    [
                        'full_name'         => $item['full_name'] ?? null,
                        'contact_number'    => $item['contact_number'] ?? null,
                        'emergency_number'  => $item['emergency_number'] ?? null,
                        'perm_address'      => $item['perm_address'] ?? null,
                        'perm_city'         => $item['perm_city'] ?? null,
                        'perm_state'        => $item['perm_state'] ?? null,
                        'perm_country'      => $item['perm_country'] ?? null,
                        'curr_address'      => $item['curr_address'] ?? null,
                        'curr_city'         => $item['curr_city'] ?? null,
                        'curr_state'        => $item['curr_state'] ?? null,
                        'curr_country'      => $item['curr_country'] ?? null,
                        'blood_group'       => $item['blood_group'] ?? null,
                        'department'        => $item['department'] ?? null,
                        'work_email'        => $item['work_email'] ?? null,
                        'location'          => $item['location'] ?? null,
                        'designation'       => $item['designation'] ?? null,
                        'created_by'        => session()->get('username'),
                        'updated_by'        => session()->get('username'),
                        'created_date'      => now(),
                        'updated_date'      => now(),
                        'active'            => 1
                    ]
                );
            }

            return response()->json(['message' => 'Staff data saved successfully']);
        } catch (\Throwable $ex) {
            Log::error('Exception in getStaffDetails', ['message' => $ex->getMessage()]);
            return response()->json(['error' => 'Something went wrong: ' . $ex->getMessage()], 500);
        }
    }
}
